--[[
    RealGPSAssets

    Adds display assets to vehicles with RealGPS specialization.

	@author: 		BayernGamers
	@date: 			13.09.2025
	@version:		1.0

	History:		v1.0 @13.09.2025 - initial implementation in FS 25
                    ------------------------------------------------------------------------------------------------------
	
	License:        Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage:
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.LOW, "RealGPSAssets.lua")

RealGPSAssets = {}
RealGPSAssets.MOD_DIR = g_currentModDirectory
RealGPSAssets.MOD_NAME = g_currentModName
RealGPSAssets.DEFAULT_VEHICLE_XML = RealGPSAssets.MOD_DIR .. "xml/defaultVehicles.xml"
RealGPSAssets.DEFAULT_VEHICLE_BASE_PATH = "defaultVehicles.vehicle(?)"
RealGPSAssets.DEFAULT_VEHICLE_SCHEMA = XMLSchema.new("realGPSDefaultVehicles")

function RealGPSAssets.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(RealGPS, specializations)
end

function RealGPSAssets.initSpecialization()
    local schema = Vehicle.xmlSchema
    schema:setXMLSpecializationType("RealGPSAssets")

    local basePath = "vehicle.realGPS.assets.asset(?)"
    schema:register(XMLValueType.STRING, basePath .. "#filename", "Filename to i3d file")
    schema:register(XMLValueType.NODE_INDEX, basePath .. "#node", "Node in external i3d file", "0|0")
    schema:register(XMLValueType.NODE_INDEX, basePath .. "#linkNode", "Link node", "0>")
    schema:register(XMLValueType.VECTOR_TRANS, basePath .. "#position", "Position", "0 0 0")
    schema:register(XMLValueType.VECTOR_ROT, basePath .. "#rotation", "Rotation", "0 0 0")
    schema:register(XMLValueType.VECTOR_SCALE, basePath .. "#scale", "Scale", "1 1 1")
    schema:register(XMLValueType.NODE_INDEX, basePath .. "#rotationNode", "Rotation node", "node")
    schema:register(XMLValueType.VECTOR_ROT, basePath .. "#rotationNodeRotation", "Rotation node rotation")
    schema:register(XMLValueType.STRING, basePath .. "#shaderParameterName", "Shader parameter name")
    schema:register(XMLValueType.VECTOR_4, basePath .. "#shaderParameter", "Shader parameter to apply")
    schema:register(XMLValueType.STRING, basePath .. "#configurationName", "Configuration name")
    schema:register(XMLValueType.INT, basePath .. "#configurationIndex", "Configuration index")

    schema = RealGPSAssets.DEFAULT_VEHICLE_SCHEMA

    schema:setXMLSpecializationType("RealGPSAssets")

    basePath = RealGPSAssets.DEFAULT_VEHICLE_BASE_PATH .. ".assets.asset(?)"

    schema:register(XMLValueType.STRING, basePath .. "#filename", "Filename to i3d file")
    schema:register(XMLValueType.NODE_INDEX, basePath .. "#node", "Node in external i3d file", "0|0")
    schema:register(XMLValueType.NODE_INDEX, basePath .. "#linkNode", "Link node", "0>")
    schema:register(XMLValueType.VECTOR_TRANS, basePath .. "#position", "Position", "0 0 0")
    schema:register(XMLValueType.VECTOR_ROT, basePath .. "#rotation", "Rotation", "0 0 0")
    schema:register(XMLValueType.VECTOR_SCALE, basePath .. "#scale", "Scale", "1 1 1")
    schema:register(XMLValueType.NODE_INDEX, basePath .. "#rotationNode", "Rotation node", "node")
    schema:register(XMLValueType.VECTOR_ROT, basePath .. "#rotationNodeRotation", "Rotation node rotation")
    schema:register(XMLValueType.STRING, basePath .. "#shaderParameterName", "Shader parameter name")
    schema:register(XMLValueType.VECTOR_4, basePath .. "#shaderParameter", "Shader parameter to apply")
    schema:register(XMLValueType.STRING, basePath .. "#configurationName", "Configuration name")
    schema:register(XMLValueType.INT, basePath .. "#configurationIndex", "Configuration index")

    schema:setXMLSpecializationType()
end

function RealGPSAssets.registerFunctions(vehicleType)
	SpecializationUtil.registerFunction(vehicleType, "onRealGPSAssetLoaded", RealGPSAssets.onRealGPSAssetLoaded)
	SpecializationUtil.registerFunction(vehicleType, "loadRealGPSAssetsFromDefaultVehicleXML", RealGPSAssets.loadRealGPSAssetsFromDefaultVehicleXML)
	SpecializationUtil.registerFunction(vehicleType, "loadRealGPSAssetsFromXML", RealGPSAssets.loadRealGPSAssetsFromXML)
end

function RealGPSAssets.registerEventListeners(vehicleType)
	SpecializationUtil.registerEventListener(vehicleType, "onPreLoad", RealGPSAssets)
	SpecializationUtil.registerEventListener(vehicleType, "onLoad", RealGPSAssets)
	SpecializationUtil.registerEventListener(vehicleType, "onDelete", RealGPSAssets)
end

function RealGPSAssets:onPreLoad(savegame)
    self.spec_realGPSAssets = {}
end

function RealGPSAssets:onLoad(savegame)
    local spec = self.spec_realGPSAssets

    spec.sharedLoadRequestIds = {}
    spec.parts = {}

    if self.configurations["realGPS"] ~= nil and self.configurations["realGPS"] == 2 then
        local isDefaultVehicle, defaultVehicleIndex = RealGPSConfig.checkIsDefaultVehicle(self.xmlFile)

        if isDefaultVehicle then
            self:loadRealGPSAssetsFromDefaultVehicleXML(defaultVehicleIndex)
        else
            self:loadRealGPSAssetsFromXML()
        end
    end
end

function RealGPSAssets:loadRealGPSAssetsFromDefaultVehicleXML(index)
    local xmlFile = XMLFile.load("defaultVehicles", RealGPSAssets.DEFAULT_VEHICLE_XML, RealGPSAssets.DEFAULT_VEHICLE_SCHEMA)
    local spec = self.spec_realGPSAssets

    if index == nil or index < 0 then
        log:printError("No valid index given to load default RealGPS for vehicle!")
        printCallstack()
        return
    end

    if xmlFile ~= nil then
        local baseKey = RealGPSAssets.DEFAULT_VEHICLE_BASE_PATH:sub(1, -4) .. string.format("(%d)", index - 1)
        xmlFile:iterate(baseKey .. ".assets.asset", function (_, key)
            local configurationName = xmlFile:getValue(key .. "#configurationName")
            local configurationIndex = xmlFile:getValue(key .. "#configurationIndex")

            if configurationName ~= nil and configurationIndex ~= 0 and self.configurations[configurationName] ~= configurationIndex then
                log:printDevInfo("Skipping RealGPS asset part '%s' as it does not match the current configuration '%s' (%d != %d)", key, configurationName, configurationIndex, self.configurations[configurationName] or -1, LoggingUtil.DEBUG_LEVELS.LOW)
                return
            end

            local filename = xmlFile:getValue(key .. "#filename")
            if filename ~= nil then
                local realGPSAsset = {}
                local isDefaultVehicle, _ = RealGPSConfig.checkIsDefaultVehicle(self.xmlFile)

                if filename:startsWith("$realGPSAssets") or isDefaultVehicle or self.baseDirectory == "" then
                    baseDirectory = RealGPSAssets.MOD_DIR
                else
                    baseDirectory = self.baseDirectory
                end

                realGPSAsset.filename = Utils.getFilename(filename, baseDirectory)
                local arguments = {
                    xmlFile = xmlFile,
                    partKey = key,
                    realGPSAsset = realGPSAsset
                }
                local sharedLoadRequestId = self:loadSubSharedI3DFile(realGPSAsset.filename, false, false, self.onRealGPSAssetLoaded, self, arguments)
                table.insert(spec.sharedLoadRequestIds, sharedLoadRequestId)
            else
                Logging.xmlWarning(xmlFile, "Missing filename for realGPS asset part '%s'", key)
            end
        end)
    end
end

function RealGPSAssets:loadRealGPSAssetsFromXML()
    local spec = self.spec_realGPSAssets

    self.xmlFile:iterate("vehicle.realGPS.assets.asset", function (_, partKey)
        local configurationName = self.xmlFile:getValue(partKey .. "#configurationName")
        local configurationIndex = self.xmlFile:getValue(partKey .. "#configurationIndex")

        if configurationName ~= nil and configurationIndex ~= 0 and self.configurations[configurationName] ~= configurationIndex then
            log:printDevInfo("Skipping RealGPS asset part '%s' as it does not match the current configuration '%s' (%d != %d)", partKey, configurationName, configurationIndex, self.configurations[configurationName] or -1, LoggingUtil.DEBUG_LEVELS.LOW)
            return
        end
        
        local filename = self.xmlFile:getValue(partKey .. "#filename")
        if filename ~= nil then
            local realGPSAsset = {}
            local baseDirectory = self.baseDirectory ~= "" and self.baseDirectory or RealGPSAssets.MOD_DIR
            realGPSAsset.filename = Utils.getFilename(filename, baseDirectory)
            local arguments = {
                xmlFile = self.xmlFile,
                partKey = partKey,
                realGPSAsset = realGPSAsset
            }
            local sharedLoadRequestId = self:loadSubSharedI3DFile(realGPSAsset.filename, false, false, self.onRealGPSAssetLoaded, self, arguments)
            table.insert(spec.sharedLoadRequestIds, sharedLoadRequestId)
        else
            Logging.xmlWarning(self.xmlFile, "Missing filename for realGPS asset part '%s'", partKey)
        end
    end)
end

function RealGPSAssets:onDelete()
    local spec = self.spec_realGPSAssets

    if spec.sharedLoadRequestIds ~= nil then
        for _, sharedLoadRequestId in ipairs(spec.sharedLoadRequestIds) do
            g_i3DManager:releaseSharedI3DFile(sharedLoadRequestId)
        end
        spec.sharedLoadRequestIds = nil
    end
end

function RealGPSAssets:onRealGPSAssetLoaded(i3dNode, failedReason, args)
    local spec = self.spec_realGPSAssets

    local xmlFile = args.xmlFile
    local partKey = args.partKey
    local realGPSAsset = args.realGPSAsset

    if i3dNode == 0 then
        Logging.xmlWarning(xmlFile, "Failed to load realGPSAsset '%s'. Unable to load i3d", partKey)
        Logging.xmlWarning(xmlFile, "Failed to load realGPSAsset '%s'. Unable to load i3d", partKey)
        return false
    end

    local node = xmlFile:getValue(partKey .. "#node", "0|0", i3dNode)
    if node == nil then
        Logging.xmlWarning(xmlFile, "Failed to load realGPSAsset '%s'. Unable to find node in loaded i3d", partKey)
        delete(i3dNode)
        return false
    end

    local linkNode = xmlFile:getValue(partKey .. "#linkNode", "0>", self.components, self.i3dMappings)
    if linkNode == nil then
        Logging.xmlWarning(xmlFile, "Failed to load realGPSAsset '%s'. Unable to find linkNode", partKey)
        delete(i3dNode)
        return false
    else
        local isReference, filename, runtimeLoaded = getReferenceInfo(linkNode)
        if isReference and runtimeLoaded then
            local xmlName = Utils.getFilenameInfo(realGPSAsset.filename, true)
            local i3dName = Utils.getFilenameInfo(filename, true)

            if xmlName ~= i3dName then
                Logging.xmlWarning(xmlFile, "RealGPSAsset '%s' loading different file from XML compared to i3D. (XML: %s vs i3D: %s)", getName(linkNode), xmlName, i3dName)
            end

            Logging.xmlWarning(xmlFile, "RealGPSAsset link node '%s' is a runtime loaded reference. Please load it either via XML or the i3D reference, but not both!", getName(linkNode))
            delete(i3dNode)
            return
        end
    end

    local x, y, z = xmlFile:getValue(partKey .. "#position")
    if x ~= nil and y ~= nil and z ~= nil then
        setTranslation(node, x, y, z)
    else
        setTranslation(node, 0, 0, 0)
    end

    local rx, ry, rz = xmlFile:getValue(partKey .. "#rotation")
    if rx ~= nil and ry ~= nil and rz ~= nil then
        setRotation(node, rx, ry, rz)
    else
        setRotation(node, 0, 0, 0)
    end

    local sx, sy, sz = xmlFile:getValue(partKey .. "#scale")
    if sx ~= nil and sy ~= nil and sz ~= nil then
        setScale(node, sx, sy, sz)
    else
        setScale(node, 1, 1, 1)
    end

    local rotationNode = xmlFile:getValue(partKey .. "#rotationNode", node, i3dNode)
    if rotationNode ~= nil and rotationNode ~= 0 then
        local rotX, rotY, rotZ = xmlFile:getValue(partKey .. "#rotationNodeRotation")
        if rotX ~= nil and rotY ~= nil and rotZ ~= nil then
            setRotation(rotationNode, rotX, rotY, rotZ)
        end
    end

    local shaderParameterName = xmlFile:getValue(partKey .. "#shaderParameterName")
    local sx, sy, sz, sw = xmlFile:getValue(partKey .. "#shaderParameter")
    if shaderParameterName ~= nil and sx ~= nil and sy ~= nil and sz ~= nil and sw ~= nil then
        setShaderParameter(node, shaderParameterName, sx, sy, sz, sw, false)
    end

    link(linkNode, node)
    delete(i3dNode)

    table.insert(spec.parts, realGPSAsset)

    return true
end