-- dirtyTireTracks.lua
-- FS25-sichere Spezialisierung für Reifendreck & Partikeleffekte
-- nutzt REA-Werte (REA_dustIntensity / REA_mudIntensity) + Wheels-Daten

dirtyTireTracks = {}
dirtyTireTracks.modName = g_currentModName
dirtyTireTracks.modDir  = g_currentModDirectory

local function dtPrint(msg)
    print(("dirtyTireTracks: %s"):format(tostring(msg)))
end

------------------------------------------------------------
-- Spezialisierungs-Basis
------------------------------------------------------------

function dirtyTireTracks.prerequisitesPresent(specializations)
    -- Keine speziellen Voraussetzungen, wird nur an Fahrzeuge mit wheels/drivable gehängt
    return true
end

function dirtyTireTracks.registerFunctions(vehicleType)
    -- Hier könnten später noch eigene Funktionen registriert werden.
end

function dirtyTireTracks.registerOverwrittenFunctions(vehicleType)
    -- Aktuell überschreiben wir keine Giants-Funktionen (sicherer für FS25).
end

function dirtyTireTracks.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad",  dirtyTireTracks)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", dirtyTireTracks)
    SpecializationUtil.registerEventListener(vehicleType, "onDelete", dirtyTireTracks)
end

------------------------------------------------------------
-- Lebenszyklus
------------------------------------------------------------

function dirtyTireTracks:onLoad(savegame)
    local spec = {}
    spec.isActive   = true
    spec.lastDust   = 0
    spec.lastMud    = 0
    spec.lastUpdate = 0

    self.spec_dirtyTireTracks = spec

    dtPrint(("Aktiv für Fahrzeug '%s'"):format(self.getName and self:getName() or "<unknown>"))
end

function dirtyTireTracks:onDelete()
    self.spec_dirtyTireTracks = nil
end

------------------------------------------------------------
-- Haupt-Update: Dreck + Partikel
------------------------------------------------------------

function dirtyTireTracks:onUpdate(dt)
    local spec = self.spec_dirtyTireTracks
    if spec == nil or not spec.isActive then
        return
    end

    -- dt in Sekunden umrechnen (FS25 liefert ms)
    local dtSec = dt * 0.001

    -- Fahrzeugweite REA-Werte (READust.lua)
    local dust = self.REA_dustIntensity or 0          -- 0..100
    local mud  = self.REA_mudIntensity  or 0          -- 0..120

    spec.lastDust = dust
    spec.lastMud  = mud

    -- Wheelspezifikation holen
    local wheelsSpec = self.spec_wheels
    local wheels = wheelsSpec and wheelsSpec.wheels
    if wheels == nil then
        return
    end

    local speedKmh = self.lastSpeedRealKmh or 0

    for _, wheel in ipairs(wheels) do
        if wheel ~= nil then
            -- Hilfs-Speicher je Rad
            wheel.spec_dirtyTireTracks = wheel.spec_dirtyTireTracks or {
                dirtLevel = 0
            }
            local ws = wheel.spec_dirtyTireTracks

            local slip = math.abs(wheel.lastSlip or 0)
            local wet  = wheel.REA_wetFactor or 0
            local sink = wheel.sink or 0

            ----------------------------------------------------------------
            -- DRECK: Reifendreck-Level berechnen (0..1)
            ----------------------------------------------------------------
            -- Basis: REA-Werte + tatsächliche Radphysik
            local dustNorm = dust / 100
            local mudNorm  = mud  / 120

            local addDirt =
                (dustNorm * 0.4 + mudNorm * 1.0) *            -- globaler REA-Dirt
                (slip * 0.6 + sink * 3.0 + 0.05) *           -- Fahrverhalten
                (wet * 1.5 + 0.3) *                          -- Feuchtigkeit verstärkt Dreck
                dtSec

            ws.dirtLevel = math.min(ws.dirtLevel + addDirt, 1.0)

            -- Reinigung: trockene Straße + Geschwindigkeit = Dreck fliegt ab
            local cleanRate = 0
            if wet < 0.05 and speedKmh > 15 then
                cleanRate = 0.03 * dtSec
            elseif wet < 0.05 and speedKmh > 5 then
                cleanRate = 0.01 * dtSec
            end

            ws.dirtLevel = math.max(ws.dirtLevel - cleanRate, 0)

            -- Giants-Dreckshader füttern, falls vorhanden
            if wheel.dirtAmount ~= nil then
                -- niemals weniger machen als Engine selbst, nur oben drauf
                local engineDirt = wheel.dirtAmount or 0
                wheel.dirtAmount = math.max(engineDirt, ws.dirtLevel)
            end

            ----------------------------------------------------------------
            -- PARTIKEL: Staub & Matsch (nur wenn passende Systeme existieren)
            ----------------------------------------------------------------
            -- HINWEIS:
            --  * Damit hier wirklich Partikel erscheinen, musst du am Fahrzeug
            --    zwei ParticleSystem-Listen setzen:
            --      self.REA_dustParticleSystems = { <ParticleSystem>, ... }
            --      self.REA_mudParticleSystems  = { <ParticleSystem>, ... }
            --  * Dieses Skript schaltet sie nur an/aus, erzeugt sie aber nicht selbst.
            ----------------------------------------------------------------

            local hasSlip = slip > 0.05
            local isMoving = speedKmh > 3

            if hasSlip and isMoving then
                -- MATSCH
                if mudNorm > dustNorm and wet > 0.3 then
                    if self.REA_mudParticleSystems ~= nil
                    and ParticleUtil ~= nil
                    and ParticleUtil.setEmittingState ~= nil then
                        for _, ps in ipairs(self.REA_mudParticleSystems) do
                            ParticleUtil.setEmittingState(ps, true)
                        end
                    end

                -- STAUB
                elseif dustNorm > 0.05 and wet < 0.3 then
                    if self.REA_dustParticleSystems ~= nil
                    and ParticleUtil ~= nil
                    and ParticleUtil.setEmittingState ~= nil then
                        for _, ps in ipairs(self.REA_dustParticleSystems) do
                            ParticleUtil.setEmittingState(ps, true)
                        end
                    end
                end
            else
                -- Fahrzeug steht oder kein Schlupf → Partikel aus
                if self.REA_mudParticleSystems ~= nil
                and ParticleUtil ~= nil
                and ParticleUtil.setEmittingState ~= nil then
                    for _, ps in ipairs(self.REA_mudParticleSystems) do
                        ParticleUtil.setEmittingState(ps, false)
                    end
                end
                if self.REA_dustParticleSystems ~= nil
                and ParticleUtil ~= nil
                and ParticleUtil.setEmittingState ~= nil then
                    for _, ps in ipairs(self.REA_dustParticleSystems) do
                        ParticleUtil.setEmittingState(ps, false)
                    end
                end
            end
        end
    end
end
